#!/bin/bash

# CCMv2 Root Certificate Update Script
# This script downloads new certificates from exposed endpoints and updates config.toml
# Supports US, EU, and AP regions

set -eo pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Configuration
CONFIG_FILE="/etc/jumpgate/config.toml"
BACKUP_DIR="/etc/jumpgate/backup"
LOG_FILE="/var/log/ccmv2_cert_auto_update.log"
TEMP_DIR="/tmp/ccmv2_cert_auto_update"

# Expected SHA256 checksum of this script (calculated with EXPECTED_SCRIPT_CHECKSUM line set to empty)
# To update after script changes: sed 's/^EXPECTED_SCRIPT_CHECKSUM="[^"]*"/EXPECTED_SCRIPT_CHECKSUM=""/' script.sh | sha256sum
EXPECTED_SCRIPT_CHECKSUM="aeaf994a065a38474d22c55a34d9c6ad6cd777c207d69b19b6ef06acb1216a1d"

get_cert_endpoint() {
    local console_domain=$1
    if [[ "$console_domain" =~ ^http:// ]]; then
        return 1
    fi
    if [[ "$console_domain" =~ ^https?:// ]]; then
        echo "${console_domain}/pki/relay-ca.pem"
    else
        echo "https://${console_domain}/pki/relay-ca.pem"
    fi
}

normalize_certificate() {
    local input_file=$1
    local output_file=$2

    # Replicate the normalization used by extract-cert.sh:
    # 1. Strip Windows carriage returns
    # 2. Remove blank/whitespace-only lines
    # 3. Ensure consistent LF-only newlines
    local temp_clean
    temp_clean=$(mktemp "$TEMP_DIR/cert_normalize_stage.XXXXXX.pem")

    sed 's/\r$//' "$input_file" | sed '/^[[:space:]]*$/d' > "$temp_clean"
    awk 'BEGIN{RS=""; ORS="\n"} {gsub(/\r/,""); print}' "$temp_clean" > "$output_file"

    rm -f "$temp_clean"
    return 0
}

get_cert_fingerprints() {
    local cert_file=$1
    local fingerprint_file=$2

    if [[ ! -f "$cert_file" ]]; then
        print_status "$RED" "Certificate file not found while extracting fingerprints: $cert_file"
        return 1
    fi

    local split_dir
    split_dir=$(mktemp -d "$TEMP_DIR/cert_split.XXXXXX")

    # Split the certificate bundle into individual PEM files
    csplit -sf "$split_dir/cert-" "$cert_file" '/-----BEGIN CERTIFICATE-----/' '{*}' >/dev/null 2>&1 || true

    local fingerprints=()
    shopt -s nullglob
    for cert in "$split_dir"/cert-*; do
        if grep -q "BEGIN CERTIFICATE" "$cert"; then
            local fingerprint
            fingerprint=$(openssl x509 -in "$cert" -noout -fingerprint -sha256 2>/dev/null | cut -d= -f2 | tr -d :)
            if [[ -n "$fingerprint" ]]; then
                fingerprints+=("$fingerprint")
            fi
        fi
    done
    shopt -u nullglob

    if [[ ${#fingerprints[@]} -eq 0 ]]; then
        print_status "$RED" "No valid certificates found while computing fingerprints for $cert_file"
        rm -rf "$split_dir"
        return 1
    fi

    printf "%s\n" "${fingerprints[@]}" | sort > "$fingerprint_file"
    rm -rf "$split_dir"
    return 0
}

get_cert_checksum() {
    local console_domain=$1
    shift
    local no_check=false
    
    # Check if --insecure flag is passed
    for arg in "$@"; do
        if [[ "$arg" == "--insecure" ]]; then
            no_check=true
            break
        fi
    done
    
    # If --insecure is set, skip checksum verification (return empty)
    if [[ "$no_check" == true ]]; then
        return 0
    fi
    
    # Extract region from console domain (e.g., "us-west-1" from "console.us-west-1.cdp.cloudera.com")
    local cdp_region
    if [[ "$console_domain" =~ https?://console\.([^.]+)\.cdp\. ]]; then
        cdp_region="${BASH_REMATCH[1]}"
    elif [[ "$console_domain" =~ console\.([^.]+)\.cdp\. ]]; then
        cdp_region="${BASH_REMATCH[1]}"
    else
        # Try to extract from domain structure
        cdp_region=$(echo "$console_domain" | awk -F'.' '{for(i=1;i<=NF;i++) if($i=="cdp") print $(i-1)}')
    fi
    
    # Return expected checksum based on region
    case "$cdp_region" in
        "us-west-1")
            echo "3a63388786ee2c8096685e740f692d60c049d38d18e025c19090cbcd1be5536b"
            ;;
        "eu-1")
            echo "15c2487e8c1ca103ce5e8ef82d66f2ab206207d99d2cd9054c9c0ebf0632f272"
            ;;
        "ap-1")
            echo "dd90bab11df14c4277bd7d1d617ace8f0cc279f3f4f965b575e02fda5ec37b15"
            ;;
        *)
            # Unknown region
            print_status "$RED" "Unknown region '${cdp_region}'. If expected, use '--insecure' to skip certificate verification" >&2
            return 1
            ;;
    esac
}

# OS Detection
get_os_version() {
    if [ -f /etc/os-release ]; then
        echo "$(grep VERSION_ID /etc/os-release | cut -d '"' -f 2 | cut -f1 -d".")"
    elif [ -f /etc/redhat-release ]; then
        echo "$(grep -oE '[0-9]+' /etc/redhat-release | head -1)"
    else
        echo "6"
    fi
}

logit() {
    local timestamp
    timestamp=$(date "+%Y-%m-%d %H:%M:%S")
    echo "[$timestamp] $*" >> "$LOG_FILE"
    chmod 600 "$LOG_FILE" 2>/dev/null || true
}

print_status() {
    local color=$1
    local message=$2
    echo -e "${color}${message}${NC}"
    logit "$message"
}

verify_script_checksum() {
    local script_path="${BASH_SOURCE[0]}"
    local calculated_checksum
    # Calculate checksum with EXPECTED_SCRIPT_CHECKSUM line set to empty to avoid circular dependency
    # This still detects tampering because modifying the checksum line changes the file structure
    calculated_checksum=$(sed 's/^EXPECTED_SCRIPT_CHECKSUM="[^"]*"/EXPECTED_SCRIPT_CHECKSUM=""/' "$script_path" 2>/dev/null | sha256sum | awk '{print $1}')
    
    if [[ -z "$EXPECTED_SCRIPT_CHECKSUM" ]]; then
        # If checksum is not set, skip verification
        return 0
    fi
    
    if [[ -z "$calculated_checksum" ]]; then
        echo -e "${RED}Error: Failed to calculate script checksum${NC}"
        exit 1
    fi
    
    if [[ "$calculated_checksum" != "$EXPECTED_SCRIPT_CHECKSUM" ]]; then
        echo -e "${RED}Error: Script integrity check failed!${NC}"
        echo -e "${RED}Expected checksum: $EXPECTED_SCRIPT_CHECKSUM${NC}"
        echo -e "${RED}Calculated checksum: $calculated_checksum${NC}"
        echo -e "${RED}The script may have been tampered with. Exiting for security.${NC}"
        exit 1
    fi
    
    # Checksum verification passed
    print_status "$GREEN" "Script integrity check passed"
}

check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_status "$RED" "This script must be run as root"
        exit 1
    fi
}

check_dependencies() {
    local missing_deps=()

    for cmd in curl openssl awk systemctl; do
        if ! command -v $cmd &> /dev/null; then
            missing_deps+=("$cmd")
        fi
    done

    if [[ ${#missing_deps[@]} -gt 0 ]]; then
        print_status "$RED" "Missing required dependencies: ${missing_deps[*]}"
        exit 1
    fi

    print_status "$GREEN" "All required dependencies are available"
}

check_config_file() {
    if [[ ! -f "$CONFIG_FILE" ]]; then
        print_status "$RED" "Config file $CONFIG_FILE not found. Please ensure CCMv2 is properly installed."
        exit 1
    fi
    print_status "$GREEN" "Found config file: $CONFIG_FILE"
}

check_service_running() {
    print_status "$BLUE" "Checking if jumpgate-agent service is running..."

    if [[ $(get_os_version) -ge 7 ]]; then
        if ! systemctl is-active --quiet jumpgate-agent.service; then
            print_status "$RED" "jumpgate-agent service is not running. Please start the service first."
            exit 1
        fi
    else
        if ! service jumpgate-agent status | grep -q "Active: active"; then
            print_status "$RED" "jumpgate-agent service is not running. Please start the service first."
            exit 1
        fi
    fi

    print_status "$GREEN" "jumpgate-agent service is running"
}

# Download certificate from endpoint
download_certificate() {
    local endpoint=$1
    local cert_file=$2
    local expected_checksum=${3:-}

    print_status "$BLUE" "Downloading certificate from: $endpoint"

    mkdir -p "$TEMP_DIR"

    local max_retries=3
    local retry_count=0

    while [[ $retry_count -lt $max_retries ]]; do
        if curl -f -s -S --connect-timeout 10 --max-time 30 -o "$cert_file" "$endpoint"; then
            if [[ -s "$cert_file" ]]; then
                print_status "$GREEN" "Certificate downloaded successfully"
                
                # Verify checksum on original download (security check)
                if [[ -n "$expected_checksum" ]]; then
                    local calculated_checksum
                    calculated_checksum=$(sha256sum "$cert_file" | awk '{print $1}')
                    print_status "$BLUE" "Downloaded certificate SHA256: $calculated_checksum"
                    
                    if [[ "$calculated_checksum" != "$expected_checksum" ]]; then
                        print_status "$RED" "Certificate checksum verification failed!"
                        print_status "$RED" "Expected checksum: $expected_checksum"
                        print_status "$RED" "Calculated checksum: $calculated_checksum"
                        return 1
                    else
                        print_status "$GREEN" "Certificate checksum verification passed"
                    fi
                else
                    print_status "$YELLOW" "Checksum verification skipped (no expected checksum for this region or --insecure flag set)"
                fi
                
                return 0
            else
                print_status "$YELLOW" "Downloaded certificate is empty, retrying..."
            fi
        else
            print_status "$YELLOW" "Download failed, attempt $((retry_count + 1)) of $max_retries"
        fi

        retry_count=$((retry_count + 1))
        sleep 2
    done

    print_status "$RED" "Failed to download certificate after $max_retries attempts"
    return 1
}

validate_certificate() {
    local cert_file=$1

    if [[ ! -f "$cert_file" ]]; then
        print_status "$RED" "Certificate file not found: $cert_file"
        return 1
    fi

    # Integrity check: SHA256 (display for reference)
    local sha256sum
    sha256sum=$(sha256sum "$cert_file" | awk '{print $1}')
    print_status "$BLUE" "SHA256 checksum: $sha256sum"

    # Certificate fingerprints (handle multiple certificates in chain)
    local cert_count
    cert_count=$(grep -c "BEGIN CERTIFICATE" "$cert_file" 2>/dev/null || echo "0")
    if [[ $cert_count -gt 1 ]]; then
        print_status "$BLUE" "Certificate chain contains $cert_count certificates:"
        local cert_num=1
        local temp_cert
        temp_cert=$(mktemp "$TEMP_DIR/cert_extract.XXXXXX.pem")
        # Split certificates and process each one
        awk '
        /-----BEGIN CERTIFICATE-----/ { 
            if (cert) print cert "-----END CERTIFICATE-----"
            cert = $0
            next
        }
        /-----END CERTIFICATE-----/ {
            cert = cert "\n" $0
            print cert
            cert = ""
            next
        }
        { if (cert) cert = cert "\n" $0 }
        END { if (cert) print cert "-----END CERTIFICATE-----" }
        ' "$cert_file" > "$temp_cert.tmp"
        while IFS= read -r cert_block || [[ -n "$cert_block" ]]; do
            if [[ -n "$cert_block" ]] && [[ "$cert_block" =~ "BEGIN CERTIFICATE" ]]; then
                echo "$cert_block" > "$temp_cert"
                local fingerprint
                fingerprint=$(openssl x509 -in "$temp_cert" -noout -fingerprint -sha256 2>/dev/null | cut -d= -f2)
                if [[ -n "$fingerprint" ]]; then
                    print_status "$BLUE" "  Certificate $cert_num fingerprint (SHA256): $fingerprint"
                    cert_num=$((cert_num + 1))
                fi
            fi
        done < "$temp_cert.tmp"
        rm -f "$temp_cert" "$temp_cert.tmp" 2>/dev/null
    else
        # Single certificate
        local fingerprint
        fingerprint=$(openssl x509 -in "$cert_file" -noout -fingerprint -sha256 2>/dev/null | cut -d= -f2)
        if [[ -n "$fingerprint" ]]; then
            print_status "$BLUE" "Certificate fingerprint (SHA256): $fingerprint"
        fi
    fi

    if openssl x509 -in "$cert_file" -text -noout >/dev/null 2>&1; then
        local expiry_date
        expiry_date=$(openssl x509 -in "$cert_file" -noout -enddate | cut -d= -f2)
        print_status "$GREEN" "Certificate format validation passed"
        print_status "$GREEN" "Certificate expires on: $expiry_date"
        return 0
    else
        print_status "$RED" "Invalid certificate format"
        return 1
    fi
}

extract_existing_certificate() {
    local config_file=$1
    local output_file=$2

    print_status "$BLUE" "Extracting existing certificate from config file..."

    local temp_raw
    temp_raw=$(mktemp "$TEMP_DIR/cert_raw.XXXXXX.pem")

    awk '
    /relayServerCertificate[[:space:]]*=/ {
        inblock = 1
        match($0, /""".*$/)
        if (RSTART > 0) {
            line = substr($0, RSTART+3)
            if (line ~ /"""/) {
                sub(/"""/, "", line)
                print line
                inblock = 0
                next
            } else {
                print line
            }
        }
        next
    }
    inblock {
        if ($0 ~ /"""/) {
            line = $0
            sub(/"""/, "", line)
            print line
            inblock = 0
            next
        }
        print
    }
    ' "$config_file" > "$temp_raw"

    if [[ ! -s "$temp_raw" ]]; then
        print_status "$YELLOW" "No existing certificate found in config file"
        rm -f "$temp_raw"
        return 1
    fi

    # Normalize using the same logic as extract-cert.sh
    normalize_certificate "$temp_raw" "$output_file"
    rm -f "$temp_raw"

    print_status "$GREEN" "Existing certificate normalized"
    return 0
}

compare_certificates() {
    local new_cert=$1
    local existing_cert=$2

    print_status "$BLUE" "Comparing certificate sets (order-independent)..."

    local new_fp_file existing_fp_file
    new_fp_file=$(mktemp "$TEMP_DIR/new_cert_fps.XXXXXX.txt")
    existing_fp_file=$(mktemp "$TEMP_DIR/existing_cert_fps.XXXXXX.txt")

    if ! get_cert_fingerprints "$new_cert" "$new_fp_file"; then
        print_status "$RED" "Failed to compute fingerprints for downloaded certificate"
        rm -f "$new_fp_file" "$existing_fp_file"
        return 1
    fi

    if ! get_cert_fingerprints "$existing_cert" "$existing_fp_file"; then
        print_status "$YELLOW" "Could not compute fingerprints for existing certificate"
        rm -f "$new_fp_file" "$existing_fp_file"
        return 1
    fi

    if diff -q "$new_fp_file" "$existing_fp_file" >/dev/null 2>&1; then
        print_status "$GREEN" "Certificate sets match (fingerprints identical)"
        rm -f "$new_fp_file" "$existing_fp_file"
        return 0
    else
        print_status "$YELLOW" "Certificate sets differ – proceeding with update"
        print_status "$BLUE" "New certificate fingerprints:"
        cat "$new_fp_file"
        print_status "$BLUE" "Existing certificate fingerprints:"
        cat "$existing_fp_file"
        rm -f "$new_fp_file" "$existing_fp_file"
        return 1
    fi
}

BACKUP_FILE_PATH=""

backup_config() {
    local timestamp
    timestamp=$(date +%Y%m%d_%H%M%S)
    local backup_file="$BACKUP_DIR/config.toml.backup.$timestamp"
    mkdir -p "$BACKUP_DIR"
    cp "$CONFIG_FILE" "$backup_file"
    chmod 600 "$backup_file" 2>/dev/null || true
    BACKUP_FILE_PATH="$backup_file"
    print_status "$GREEN" "Backup created: $backup_file"
}

restart_service() {
    if [[ $(get_os_version) -ge 7 ]]; then
        systemctl restart jumpgate-agent.service
    else
        service jumpgate-agent restart
    fi
}

check_service_active() {
    if [[ $(get_os_version) -ge 7 ]]; then
        systemctl is-active --quiet jumpgate-agent.service
    else
        service jumpgate-agent status | grep -q "Active: active"
    fi
}

get_service_status() {
    if [[ $(get_os_version) -ge 7 ]]; then
        systemctl status jumpgate-agent.service --no-pager
    else
        service jumpgate-agent status
    fi
}

restart_agent() {
    print_status "$YELLOW" "Restarting jumpgate-agent service..."

    restart_service
    sleep 3

    if check_service_active; then
        print_status "$GREEN" "jumpgate-agent service restarted successfully"
        print_status "$BLUE" "Service status:"
        get_service_status | head -8
        return 0
    else
        print_status "$RED" "Failed to restart jumpgate-agent service"
        get_service_status | head -8
        return 1
    fi
}

update_config() {
    local new_cert_file=$1

    print_status "$YELLOW" "Updating config.toml with new certificate..."

    local temp_config
    temp_config=$(mktemp "$TEMP_DIR/config.toml.tmp.XXXXXX")

    awk -v cert_file="$new_cert_file" '
    BEGIN {
        in_cert_section = 0
        cert_updated = 0
        new_cert = ""
        cert_line_count = 0
        while ((getline line < cert_file) > 0) {
            cert_line_count++
            new_cert = new_cert (cert_line_count == 1 ? "" : "\n") line
        }
        close(cert_file)
        sub(/\n+$/, "", new_cert)
    }
    /^relayServerCertificate = / {
        printf "relayServerCertificate = \"\"\"%s\"\"\"\n", new_cert
        in_cert_section = 1
        cert_updated = 1
        next
    }
    in_cert_section && /\"\"\"$/ {
        in_cert_section = 0
        next
    }
    !in_cert_section {
        print
    }
    END {
        if (!cert_updated) {
            print "ERROR: relayServerCertificate field not found" > "/dev/stderr"
            exit 1
        }
    }' "$CONFIG_FILE" > "$temp_config"

    mv "$temp_config" "$CONFIG_FILE"
    chmod 644 "$CONFIG_FILE"

    print_status "$GREEN" "Config.toml updated successfully"
}

validate_agent_service() {
    print_status "$YELLOW" "Validating jumpgate-agent service..."
    sleep 10

    if ! check_service_active; then
        print_status "$RED" "jumpgate-agent service is not running"
        return 1
    fi

    local error_count=0
    if [[ $(get_os_version) -ge 7 ]]; then
        error_count=$(journalctl -u jumpgate-agent.service --since "1 minute ago" | grep -i error | wc -l)
    else
        error_count=$(tail -20 /var/log/jumpgate/out.log 2>/dev/null | grep -i error | wc -l)
    fi

    if [[ $error_count -gt 0 ]]; then
        print_status "$RED" "Found $error_count errors in service logs"
        return 1
    fi

    print_status "$GREEN" "Service validation passed"
    return 0
}

rollback() {
    local backup_file=$1

    print_status "$YELLOW" "Rolling back to previous configuration..."

    if [[ -f "$backup_file" ]]; then
        cp "$backup_file" "$CONFIG_FILE"
        chmod 644 "$CONFIG_FILE"

        if restart_agent; then
            print_status "$GREEN" "Rollback completed successfully"
        else
            print_status "$RED" "Rollback failed - service could not be restarted"
        fi
    else
        print_status "$RED" "Backup file not found: $backup_file"
    fi
}

list_backups() {
    print_status "$BLUE" "Available backups:"

    if [[ ! -d "$BACKUP_DIR" ]] || [[ -z "$(ls -A "$BACKUP_DIR" 2>/dev/null)" ]]; then
        print_status "$YELLOW" "No backups found"
        return 0
    fi

    ls -lh "$BACKUP_DIR"/config.toml.backup.* 2>/dev/null
}

show_help() {
    cat << EOF
CCMv2 Certificate Auto-Update Script
====================================

Usage:
  $0 update <console_domain>  # Update certificate for specified console_domain
  $0 list-backups             # List available backups
  $0 help                     # Show this help


Examples:
  $0 update https://console.us-west-1.cdp.cloudera.com  # Update with automatic checksum verification
  $0 update https://console.dps.mow-dev.cloudera.com --insecure  # Update without checksum verification
  $0 list-backups

Certificate Checksum Verification:
  The script automatically determines the expected certificate checksum based on
  the region extracted from the console domain. Supported regions:
  - us-west-1, eu-1, ap-1 (automatic checksum verification)
  - Other regions: Use --insecure flag to skip verification
  
  If an unknown region is detected, the script will fail unless --insecure
  is explicitly provided to acknowledge skipping verification for unknown regions.

Prerequisites:
  - Root access required
  - Internet connectivity to CCM endpoints
  - Required tools: curl, openssl, awk, systemctl

Note:
  This script automatically downloads certificates from Cloudera CCM endpoints
  and updates the jumpgate-agent configuration.
  Console domain parameter is mandatory.
EOF
}


cleanup() {
    if [[ -d "$TEMP_DIR" ]]; then
        rm -rf "$TEMP_DIR"
    fi
}

main() {
    # Verify script integrity first (before any other operations)
    verify_script_checksum
    
    local command=${1:-}
    local console_domain=${2:-}
    shift 2 2>/dev/null || shift 1 2>/dev/null || true
    local remaining_args=("$@")

    case "$command" in
        "update")
            if [[ -z "$console_domain" ]]; then
                print_status "$RED" "Error: Console domain parameter is required"
                echo ""
                show_help
                exit 1
            fi
            
            # Get expected checksum from region or arguments
            local expected_checksum
            if ! expected_checksum=$(get_cert_checksum "$console_domain" "${remaining_args[@]}"); then
                # get_cert_checksum failed (unknown region), error message already printed
                exit 1
            fi
            ;;
        "list-backups")
            list_backups
            exit 0
            ;;
        "help"|"-h"|"--help")
            show_help
            exit 0
            ;;
        *)
            show_help
            exit 1
            ;;
    esac

    # Create temp directory before using mktemp
    mkdir -p "$TEMP_DIR"

    local cert_file
    cert_file=$(mktemp "$TEMP_DIR/new_ccm_certificate.XXXXXX.pem")

    trap cleanup EXIT

    print_status "$BLUE" "=========================================="
    print_status "$BLUE" "CCMv2 Certificate Auto-Update Starting"
    print_status "$BLUE" "=========================================="

    check_root
    check_dependencies
    check_service_running
    check_config_file

    # Download certificate
    local endpoint
    if ! endpoint=$(get_cert_endpoint "$console_domain"); then
        print_status "$RED" "Error: HTTP is not allowed for certificate download. Use HTTPS instead."
        exit 1
    fi
    if ! download_certificate "$endpoint" "$cert_file" "$expected_checksum"; then
        print_status "$RED" "Certificate download or checksum verification failed"
        exit 1
    fi

    # Validate certificate (on original downloaded cert)
    if ! validate_certificate "$cert_file"; then
        print_status "$RED" "Certificate validation failed"
        exit 1
    fi

    # Check if existing certificate is already up to date
    # Step 1: Normalize the downloaded certificate for comparison
    local normalized_new_cert
    normalized_new_cert=$(mktemp "$TEMP_DIR/new_cert_normalized.XXXXXX.pem")
    normalize_certificate "$cert_file" "$normalized_new_cert"
    print_status "$GREEN" "Downloaded certificate normalized for comparison"
    
    # Step 2: Extract and normalize existing certificate from config
    local normalized_existing_cert
    normalized_existing_cert=$(mktemp "$TEMP_DIR/existing_cert_normalized.XXXXXX.pem")
    
    if extract_existing_certificate "$CONFIG_FILE" "$normalized_existing_cert"; then
        # Step 3: Compare the normalized versions
        if compare_certificates "$normalized_new_cert" "$normalized_existing_cert"; then
            print_status "$GREEN" "=========================================="
            print_status "$GREEN" "All certificates are up to date!"
            print_status "$GREEN" "No update needed."
            print_status "$GREEN" "=========================================="
            exit 0
        fi
    else
        print_status "$YELLOW" "Could not extract existing certificate for comparison, proceeding with update..."
    fi

    # Create backup
    backup_config

    # Update configuration with the NORMALIZED certificate (so it's consistent for future comparisons)
    if ! update_config "$normalized_new_cert"; then
        print_status "$RED" "Config update failed"
        rollback "$BACKUP_FILE_PATH"
        exit 1
    fi

    # Restart agent
    if ! restart_agent; then
        print_status "$RED" "Failed to restart agent"
        rollback "$BACKUP_FILE_PATH"
        exit 1
    fi

    # Validate service
    if ! validate_agent_service; then
        print_status "$RED" "Service validation failed"
        rollback "$BACKUP_FILE_PATH"
        exit 1
    fi

    # Success
    print_status "$GREEN" "=========================================="
    print_status "$GREEN" "Certificate update completed successfully!"
    print_status "$GREEN" "Domain: $console_domain"
    print_status "$GREEN" "Backup: $BACKUP_FILE_PATH"
    print_status "$GREEN" "=========================================="
}

if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi
